/*:
 * @target MZ
 * @plugindesc v1.2 ムービー音量を調整（オプション非表示・再生時に確実反映）
 * @author HS
 *
 * @command setVolume
 * @text ムービー音量を設定
 * @arg value
 * @type number
 * @min 0
 * @max 100
 * @default 100
 *
 * @command fadeTo
 * @text ムービー音量をフェード
 * @desc 現在のムービー音量を指定値までミリ秒でフェード
 * @arg value
 * @type number
 * @min 0
 * @max 100
 * @default 100
 * @arg durationMs
 * @text 期間(ミリ秒)
 * @type number
 * @min 0
 * @default 500
 */
(() => {
  const pluginName = "HS_MovieVolumeMZ";

  // -----------------------------
  // 保存のみ（オプションには項目を出さない）
  // -----------------------------
  ConfigManager.movieVolume = 100;
  const _make = ConfigManager.makeData;
  ConfigManager.makeData = function () {
    const c = _make.call(this);
    c.movieVolume = this.movieVolume;
    return c;
  };
  const _apply = ConfigManager.applyData;
  ConfigManager.applyData = function (c) {
    _apply.call(this, c);
    this.movieVolume = this.readVolume(c, "movieVolume");
  };

  // -----------------------------
  // Core
  // -----------------------------
  function desiredVolume01() {
    return Math.max(0, Math.min(1, (ConfigManager.movieVolume ?? 100) / 100));
  }
  function applyMovieVolume() {
    const el = Video && Video._element;
    if (!el) return;
    const v = desiredVolume01();
    if (el.volume !== v) el.volume = v;
    el.muted = v <= 0;
  }
  function hookVideoElement(el) {
    if (!el || el._hs_mv_hooked) return;
    el._hs_mv_hooked = true;
    el.addEventListener("loadedmetadata", applyMovieVolume);
    el.addEventListener("play", () => setTimeout(applyMovieVolume, 0));
    el.addEventListener("playing", applyMovieVolume);
  }

  // 要素生成時＆再生時に強制適用（複数フレーム念押し）
  const _create = Video._createElement;
  Video._createElement = function () {
    _create.call(this);
    hookVideoElement(this._element);
    applyMovieVolume();
  };

  const _play = Video.play;
  Video.play = function (src) {
    _play.call(this, src);
    let tries = 0;
    const maxTries = 10; // 約10フレーム分
    const tick = () => {
      hookVideoElement(this._element);
      applyMovieVolume();
      if (tries++ < maxTries) requestAnimationFrame(tick);
    };
    requestAnimationFrame(tick);
  };

  // -----------------------------
  // Commands
  // -----------------------------
  PluginManager.registerCommand(pluginName, "setVolume", (args) => {
    const n = Math.round(Math.max(0, Math.min(100, Number(args.value) || 0)));
    ConfigManager.movieVolume = n;
    ConfigManager.save();
    applyMovieVolume();
  });

  PluginManager.registerCommand(pluginName, "fadeTo", (args) => {
    const el = Video && Video._element;
    const target = Math.max(0, Math.min(100, Number(args.value) || 0)) / 100;
    const dur = Math.max(0, Number(args.durationMs) || 0);
    ConfigManager.movieVolume = Math.round(target * 100);
    ConfigManager.save();
    if (!el || dur === 0) return applyMovieVolume();
    const start = performance.now();
    const v0 = el.volume ?? desiredVolume01();
    const anim = (t0) => {
      const t = performance.now() - t0;
      const k = Math.min(1, t / dur);
      el.volume = v0 + (target - v0) * k;
      el.muted = el.volume <= 0;
      if (k < 1) requestAnimationFrame(() => anim(t0));
    };
    requestAnimationFrame(() => anim(start));
  });
})();


